/*---------------------------------------------------------------------------*\

    FILE....: VPBREG.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 12/9/99

    Helper functions to set up the configuration database known as the DSP 
    registry.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "vpbreg.h"
#include "wobbly.h"
#include "mess.h"
#include "vpbapi.h"
#include "apifunc.h"
#include "generic.h"
#include "dspfifo.h"
#include "verbose.h"
#include "vpbdial.h"

// Compile-time constants --------------------------------------------

#define	NUM_CARDS		1     // def number of VPBs in this PC
#define	BASE_ADDRESS	        0x310 // def base address of VPB

#define	NUM_CHANNELS_VPB4       4     // number of channels in VPB4, V4PCI
#define	NUM_CHANNELS_VPB8L      8     // number of channels in VPB8L
#define	SIZE_MESS_Q		256   // size of message FIFOs (16-bit words)

// DR 24/9/02 - note changing any of these 3 below may kill host based EC
// code in dspfifo.cpp may also need mods

#define	SIZE_RX_Q		800   // size of Rx (up) signal FIFOs
#define	SIZE_TX_Q		800   // size of Tx (down) signal FIFOs
#define	LENGTH_FRAME	        160   // samples in each processing frame

// device driver types

#define	DD_ISA_PORT		0     // ISA port I/O only device driver
#define DD_ISA_RELAY            1     // ISA relay FIFO device driver
#define DD_PCI                  2     // PCI driver
#define DD_V12PCI               3     // V12PCI driver
#define DD_V6PCI                3     // V6PCI driver
     
#define	SIZE_RELAY_BUF	16000	      // size of relay buffer in device driver

// default gains (VPB4 & V4PCI & V12PCI)

#define DEF_RECORD_GAIN_ISA      0.0
#define DEF_PLAY_GAIN_ISA        0.0
#define DEF_RECORD_GAIN_PCI      6.0
#define DEF_PLAY_GAIN_PCI        7.0
#define DEF_RECORD_GAIN_V12PCI   3.0
#define DEF_PLAY_GAIN_V12PCI     3.0

// default codec balance registers

#define DEF_BAL1                 0xc7
#define DEF_BAL2                 0x00
#define DEF_BAL3                 0x00

// path and name of firmware file

#define	FIRMWARE_FILE_ISA	"/etc/vpb/vpbmain_isa.out"	
#define	FIRMWARE_FILE_ISA_VPB8L	"/etc/vpb/vlcmain.out"	
#define	FIRMWARE_FILE_PCI	"/etc/vpb/vpbmain_pci.out"	

/*-------------------------------------------------------------------------*\

			     STATICS

\*-------------------------------------------------------------------------*/

static int vpb_num_cards = NUM_CARDS;
static int def_base[1] = {BASE_ADDRESS};
static int *vpb_base = def_base;
static char *vpb_firmware_file = FIRMWARE_FILE_PCI;
static int vpb_model = VPB_V4PCI;
static int defaultsused = 1;

static int def_bal1 = DEF_BAL1;
static int def_bal2 = DEF_BAL2;
static int def_bal3 = DEF_BAL3;

static USHORT def_hp = 0;
static float  defPlayGain = DEF_PLAY_GAIN_PCI;
static float  defRecordGain = DEF_RECORD_GAIN_PCI;

static USHORT def_hostec = 0;
static int def_echo_en = 0;
static USHORT def_dighyb = 0;
static char *def_dighybfile = NULL;
 
/*-------------------------------------------------------------------------*\

			      FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_config
	AUTHOR...: David Rowe
	DATE.....: 24/8/00

	Set installation specific config info, call before vpb_open().  If
	this function is called, it overrides any other settings (for example
	compile-time defaults or env vars).

\*--------------------------------------------------------------------------*/

int WINAPI vpb_config(int num_cards, int *bases, char *firmware_file, int model) {

	assert((num_cards > 0) && (num_cards <MAX_VPB));
	assert(bases != NULL);
	assert(firmware_file != NULL);

	vpb_num_cards = num_cards;
	vpb_base = bases;
	vpb_firmware_file = firmware_file;
	vpb_model = model;

	defaultsused = 0;

	return VPB_OK;
}

/*-------------------------------------------------------------------------*\

			      CLASS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: ~VPBRegister
	AUTHOR...: John Kostogiannis
	DATE.....: 18/11/97

	Closes the VPB registry.

\*--------------------------------------------------------------------------*/

VPBRegister::~VPBRegister() {
	
	// free allocated VPB registry memory 

	assert(reg != NULL);
	free(reg);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: VPBRegister
	AUTHOR...: David Rowe
	DATE.....: 12/9/99

	Creates the VPB registry using compile-time constants.

\*--------------------------------------------------------------------------*/

VPBRegister::VPBRegister(USHORT *num)
//      USHORT *num;    number of VPBs
{
	*num = vpb_num_cards;
        int    hybdefs_overwrite;
   
	VPBREG *preg; // ptr to current VPB registry (one per VPB card)
	USHORT i,j;
	
	// validate arguments 

	assert(num != NULL);

	// attempt to autodetect hardware card type (can be overridden
	// by env variables

	#ifndef __FreeBSD__
	if (defaultsused) {
	  vpb_model =  GenericGetCardType();
	  mprintf("vpb_model = %d\n", vpb_model);
	}
	#endif
	
	// check for environment variables that override defaults

	// if defaults have already been overwritten by call to vpb_config
	// ignore env vars
	if (defaultsused) {
	  char *new_firmware, *new_model, *new_base, *new_bal1, *new_bal2;
	  char *new_bal3, *new_RecordGain,*new_PlayGain;
	  char *new_hostec, *new_dighyb, *new_verbose, *new_dtmf;

	  // environment variable controlled message logging

	  new_verbose = getenv("VPB_VERBOSE");
	  if (new_verbose) {
		  // if argument == 1, just switch on console logging
		  if (strcmp(new_verbose, "1") == 0) {
			  verbose(1);
		  }
		  else {
			  // otherwise use file logging
			  verbose_file(1, new_verbose);
		  }
	  }

	  new_model = getenv("VPB_MODEL");
	  if (new_model) {
	    if (strcmp(new_model, "VPB4")==0) {
	      vpb_model = VPB_VPB4;
	      vpb_firmware_file = FIRMWARE_FILE_ISA;      
	    }
	    if (strcmp(new_model, "VPB8L")==0) {
	      vpb_model = VPB_VPB8L;
	      vpb_firmware_file = FIRMWARE_FILE_ISA_VPB8L;      
	    }
	    if (strcmp(new_model, "V4PCI")==0) {
	      vpb_model = VPB_V4PCI;
	      vpb_firmware_file = FIRMWARE_FILE_PCI;      
	    }
	    if (strcmp(new_model, "V4LOG")==0) {
	      vpb_model = VPB_V4LOG;
	      vpb_firmware_file = FIRMWARE_FILE_PCI;      
	    }
	    if (strcmp(new_model, "V12PCI")==0) {
	      vpb_model = VPB_V12PCI;
	      vpb_firmware_file = "";      
	    }
	    if (strcmp(new_model, "V6PCI")==0) {
	      vpb_model = VPB_V12PCI;
//	      vpb_model = VPB_V6PCI;
	      vpb_firmware_file = "";      
	    }
	  }

	  new_firmware = getenv("VPB_FIRMWARE");
	  if (new_firmware) {
	    vpb_firmware_file = new_firmware;
	  }

	  new_base = getenv("VPB_BASE");
	  if (new_base) {
	    int base;
	    sscanf(new_base, "%x", &base);
	    def_base[0] = base;
	  }

	  // codec balance values specified by env vars

	  hybdefs_overwrite = 0;
	  new_bal1 = getenv("VPB_BAL1");
	  if (new_bal1) {
	    sscanf(new_bal1, "%x", &def_bal1);
	    mprintf("new bal1 = 0x%x\n", def_bal1);
	    hybdefs_overwrite = 1;
	  }
	  new_bal2 = getenv("VPB_BAL2");
	  if (new_bal2) {
	    sscanf(new_bal2, "%x", &def_bal2);
	    mprintf("new bal2 = 0x%x\n", def_bal2);
	    hybdefs_overwrite = 1;
	  }
	  new_bal3 = getenv("VPB_BAL3");
	  if (new_bal3) {
	    sscanf(new_bal3, "%x", &def_bal3);
	    mprintf("new bal3 = 0x%x\n", def_bal3);
	    hybdefs_overwrite = 1;
	  }

	  // optional HP filter on Tx and RX
	  if (getenv("VPB_HP")) {
		  def_hp = 1;
	  }

	  // Record and Play gains
	  new_RecordGain = getenv("VPB_RG");
	  if (new_RecordGain) {
	    sscanf(new_RecordGain, "%f", &defRecordGain);
	  }
	  new_PlayGain = getenv("VPB_PG");
	  if (new_PlayGain) {
	    sscanf(new_PlayGain, "%f", &defPlayGain);
	  }

	  // host echo canceller
	  new_hostec = getenv("VPB_HOSTEC");
	  if (new_hostec) {
	    def_hostec = DSPFIFO_ECHO;
	    sscanf(new_hostec, "%d", &def_echo_en);
	    mprintf("Host echo canc, enable = %d\n", def_echo_en);
	  }

	  // digital hybrid balance filter, note that we must enable hostec
	  // so that dspfifo ECHO_THREADs run
	  new_dighyb = getenv("VPB_DIGHYB");
	  if (new_dighyb) {
		  def_dighyb = 1;
	          def_hostec = DSPFIFO_ECHO;
		  def_dighybfile = new_dighyb;
		  mprintf("digital hybrid balance filter enabled\n");
		  mprintf("expected impulse responsefiles = %s0 %s1 ....\n", 
			  def_dighybfile);
	  }

	  // env variable for DTMF length in ms

	  new_dtmf = getenv("VPB_DTMF");
	  if (new_dtmf) {
		  vpbdial_change_dtmf_length(atoi(new_dtmf), atoi(new_dtmf));
		  mprintf("new dtmf on/off time = %s\n", new_dtmf);
      	  }

	} // end env variable scanning..............

	// auto-detect number of cards in system..........

	#ifdef __FreeBSD__
	*num = vpb_num_cards;
	#else  
	if (defaultsused) {
		if ((vpb_model == VPB_V4PCI)||(vpb_model == VPB_V4LOG)
		    || (vpb_model == VPB_V12PCI)|| (vpb_model == VPB_V6PCI)) {
			// auto-determine number of cards if PCI
			int numCards;

			numCards = GenericGetNumCards(vpb_model);
			*num = numCards;			
		}
	}
	else {
		// hard code number of cards if defaults over-ridden
		*num = vpb_num_cards;
	} 
	#endif
	
	if (*num==0)
	  throw Wobbly(VPBREG_NO_VPB_DEVICES_DETECTED);

	// Allocate memory for VPB DSP registry
	reg = (VPBREG*)malloc(sizeof(VPBREG)*(*num));
	assert(reg != NULL);

	mprintf("num = %d model = %d\n", *num, vpb_model);
	for(i=0; i<*num; i++) {
		// set up VPB Reg info
		preg = &reg[i];

		switch(vpb_model) {
		case VPB_VPB4:
			preg->numch = 4;
			preg->model = VPB_VPB4;
			preg->base = (USHORT)vpb_base[i];
			break;
		case VPB_V4PCI:
			preg->numch = 4;
			preg->model = VPB_VPB4;
			preg->base = i;
			break;
		case VPB_V4LOG:
			preg->numch = 4;
			preg->model = VPB_VPB4;
			preg->base = i;
			break;
		case VPB_VPB8L:
			preg->numch = 8;
			preg->model = VPB_VPB8L;
			preg->base = (USHORT)vpb_base[i];
			break;
		case VPB_V12PCI:
			preg->numch = 12;
			preg->model = VPB_V12PCI;
			preg->base = i;
			break;
		case VPB_V6PCI:
			preg->numch = 12;	//first 6 ports empty
			preg->model = VPB_V12PCI;
			preg->base = i;
			// OS_BASE=6;
			break;
		default:
			assert(0);
			break;
		}
		preg->szmess = SIZE_MESS_Q;
		for(j=0; j<preg->numch; j++) {
			preg->szrxdf[j] = SIZE_RX_Q;
			preg->sztxdf[j] = SIZE_RX_Q;
		}
		preg->lsf = LENGTH_FRAME;

		switch(vpb_model) {
		case VPB_VPB4:
			preg->ddmodel = DD_ISA_PORT;
			preg->defRecordGain = DEF_RECORD_GAIN_ISA;
			preg->defPlayGain = DEF_PLAY_GAIN_ISA;
			break;
		case VPB_V4PCI:
			preg->ddmodel = DD_PCI;
			preg->defRecordGain = defRecordGain;
			preg->defPlayGain = defPlayGain;
			break;
		case VPB_V4LOG:
			preg->ddmodel = DD_PCI;
			preg->defRecordGain = defRecordGain;
			preg->defPlayGain = defPlayGain;
			break;
		case VPB_VPB8L:
			preg->ddmodel = DD_ISA_PORT;
			break;
		case VPB_V12PCI:
			preg->ddmodel = DD_V12PCI;
			preg->defRecordGain = DEF_RECORD_GAIN_V12PCI;
			preg->defPlayGain = DEF_PLAY_GAIN_V12PCI;
			break;
		case VPB_V6PCI:
			preg->ddmodel = DD_V12PCI;
			preg->defRecordGain = DEF_RECORD_GAIN_V12PCI;
			preg->defPlayGain = DEF_PLAY_GAIN_V12PCI;
			break;
		default:
			assert(0);
			break;
		}
		preg->szRelayBuf = 0; // unused
		
		preg->hybdefs_overwrite = hybdefs_overwrite;
		preg->defbal1 = def_bal1;
		preg->defbal2 = def_bal2;
		preg->defbal3 = def_bal3;
		preg->hp_enable = def_hp;
		preg->echo_mode = def_hostec;
		preg->echo_en = def_echo_en;
		preg->dighyb = def_dighyb;
		preg->dighybfile = def_dighybfile;

		// init ptrs to PC copy of FIFO state variables
		preg->dnmess = NULL;		
		preg->upmess = NULL;			
		for(j=0; j<MAXCH; j++) {
			preg->rxdf[j] = NULL;		
			preg->txdf[j] = NULL;		
		}
		// set up firmware path and file name
		strcpy(preg->firm, vpb_firmware_file);
	}

}

int vpbreg_get_model() {
	return vpb_model;
}

void vpbreg_load_default_tones(int numch)
{
	// defaults for tone detectors should be loaded here
}

